#include <cstdlib>
#include <iostream>
#include <cmath>

#include "utils.h"
#include "integrate_ode.h"

using namespace std;

/*
    (c) 2005 by Jan Krieger  -- jan@jkrieger.de - www.jkrieger.de
    
    Dieses Programm integriert das 2-Krper-Problem
      r' = v
      v' = -r/r^3
    
*/

#define dimension 4


double Omega(double* y) {
  double r=sqrt(y[1]*y[1]+y[2]*y[2]);   
  return 1/r;     
}

void f(double* yout, double* y, double x) {
  double r=sqrt(y[1]*y[1]+y[2]*y[2]);   
  yout[1]=y[3];
  yout[2]=y[4];
  yout[3]=-y[1]/(r*r*r);
  yout[4]=-y[2]/(r*r*r);
}


void leapfrogttl_step(double* yout, double* y, double& t, double& W, double h,
	void (*f)(double*, double*, double)) {
  long size=v_getsize(yout);
  double* rhalf=vector(size);
  double* a=vector(size);
  double thalf;
  double om, r;
  v_copy(yout, y);

  // r_{1/2} = r_0 + h/2*v
  rhalf[1]=y[1]+h/2.0*y[3]/W;     
  rhalf[2]=y[2]+h/2.0*y[4]/W;   
  rhalf[3]=0;  
  rhalf[4]=0;  
  thalf=t+h/(2.0*W);
  
  // v_1 = v_0 + h*F(r_{1/2})
  f(a, rhalf, t+h/2);
  r=v_abs(rhalf);
  om=Omega(rhalf);
  yout[3]=y[3]+h*a[3]/om;
  yout[4]=y[4]+h*a[4]/om;
  W=W+h/(2*om*r*r*r)*((y[3]+yout[3])*rhalf[1] + (y[4]+yout[4])*rhalf[2]);
  
  // r_1 = r_{1/2} + h/2 * v_1
  yout[1]=rhalf[1]+h/2.0*yout[3];
  yout[2]=rhalf[2]+h/2.0*yout[4];
  t=thalf+h/(2.0*W);
  
  v_free(rhalf);
  v_free(a);
}

void angular_momentum(double* j, double* pos) {
     double* r=vector(3);
     double* p=vector(3);
     
     r[1]=pos[1]; r[2]=pos[2]; r[3]=0;
     p[1]=pos[3]; p[2]=pos[4]; p[3]=0;
     
     v_cross(j, r, p);
     v_free(r);
     v_free(p);
}

void runge_lenz(double* e, double* pos) {
     double* r=vector(3);
     double* p=vector(3);
     double* j=vector(3);
     
     r[1]=pos[1]; r[2]=pos[2]; r[3]=0;
     p[1]=pos[3]; p[2]=pos[4]; p[3]=0;
     
     angular_momentum(j, pos);
     v_cross(e, p, j);
     v_normalize(r);
     v_multconstself(r, -1.0);
     v_addself(e,r);
     
     v_free(r);
     v_free(p);
     v_free(j);
}

double energy(double* pos) {
  return 0.5*(pos[3]*pos[3]+pos[4]*pos[4])-1/sqrt(pos[1]*pos[1]+pos[2]*pos[2]);
}

int main(int argc, char *argv[])
{
    double* y0=vector(dimension);
    double* yout2=vector(dimension);
    double* yout4=vector(dimension);
    double* youte=vector(dimension);
    double* youtl=vector(dimension);
    double h=0.001;
    double tmax=100;
    double E0, E, W;
    int count=0;
    int cmax=20;
    y0[1]=1.0;
    y0[2]=0;
    y0[3]=0;
    y0[4]=1.05;
    W=1;
    v_copy(yout2, y0);
    v_copy(yout4, y0);
    v_copy(youte, y0);
    v_copy(youtl, y0);
    E0=energy(y0);
    cout.precision(15);
    for(double t=0; t<=tmax; ) {
      count++;
      if (count>cmax) cout<<t<<",\t";

      v_copy(y0, youtl);
      leapfrogttl_step(youtl, y0, t, W, h, f);
      E=energy(youtl);
      if (count>cmax) cout <<youtl[1]<<",\t"<<youtl[2]<<",\t"<<abs((E-E0)/E0)<<",\t";
       
      if (count>cmax) {cout<<endl; count=0;}
    }                
    v_free(y0);
    v_free(yout2);
    v_free(yout4);
    v_free(youte);
    v_free(youtl);
    return EXIT_SUCCESS;
}
