unit JKElecControls;

interface

{-------------------------------------------------------------------------------
Author:      Jan Krieger, Ismaning (Munich), Germany
             jan@jkrieger.de
             http://www.jkrieger.de/
Date:        June 10./11. 2000
Name:        TJKMeter
Version:     1.0
Copyright:   (c) 2000 by Jan W. Krieger. All rights reserved.
Description: This Component shows an analog meter with a liner scale
             Max/Min represent its range
             Position represents the indicator's current position
                      (values bigger than Max are displayed as Max)
                      (values smaller than Min are displayed as Min)
             many more properties to customize the meter!

             procedure PaintHousing;    paints the housing of the meter
             procedure PaintScale;      paints the scale
             procedure PaintIndicator;  adds the Indicator
             => to prevent flickering, everything is painted into FTempBitmap
                and FTempBitMap is finally copied (procedure TCanvas.Draw) into
                the components own Canvas!
Status:      This Component is FREEWARE, so no support can be granted
             Everybody may feel free to copy and use it.
             If you alter the sourcecode, it would be kind to send me the
             new source!

--------------------------------------------------------------------------------
                   This comment may not be deleted !!!
--------------------------------------------------------------------------------}

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, extCtrls;

type
  TJKMeter = class(TGraphicControl)
  private
    { Private-Deklarationen }
    FMax, FMin, FPosition:extended;
    FCaptionDistance, FBigMarkDistance, FSmallMarkDistance:extended;
    FShowUnitName: boolean;
    FUnitName: string;
    FCaptionFont, FUnitFont:TFont;
    FPosColor, FNegColor, FZeroColor, FIndicatorColor, FBorderColor, FBackGroundColor:TColor;
    FIndicatorWidth, FBorderWidth:longint;
    FSmallMarkLength, FBigMarkLength, FSmallMarkWidth, FBigMarkWidth:longint;
    FTempBitMap:TBitMap;

    procedure ClearCanvas;
    procedure PaintHousing;
    procedure PaintScale;
    procedure PaintIndicator;

    function getMax:extended;
    procedure setMax(value: extended);
    function getMin:extended;
    procedure setMin(value: extended);
    function getPosition:extended;
    procedure setPosition(value: extended);
    function getCaptionDistance:extended;
    procedure setCaptionDistance(value: extended);
    function getBigMarkDistance:extended;
    procedure setBigMarkDistance(value: extended);
    function getSmallMarkDistance:extended;
    procedure setSmallMarkDistance(value: extended);
    function getShowUnitName:boolean;
    procedure setShowUnitName(value: boolean);
    function getUnitName:string;
    procedure setUnitName(value: string);
    function getCaptionFont:TFont;
    procedure setCaptionFont(value:TFont);
    function getUnitFont:TFont;
    procedure setUnitFont(value:TFont);
    function getPositiveColor:TColor;
    procedure setPositiveColor(value: TColor);
    function getNegativeColor:TColor;
    procedure detNegativeColor(value: TColor);
    function getZeroColor:TColor;
    procedure setZeroColor(value: TColor);
    function getIndicatorColor:TColor;
    procedure setIndicatorColor(value: TColor);
    function getBorderColor:TColor;
    procedure setBorderColor(value: TColor);
    function getBackgroundColor:TColor;
    procedure setBackgroundColor(value: TColor);
    function getIndicatorWidth:longint;
    procedure setIndicatorWidth(value: longint);
    function getBorderWidth:longint;
    procedure setBorderWidth(value: longint);
    function getSmallMarkLength:longint;
    procedure setSmallMarkLength(value: longint);
    function getBigMarkLength:longint;
    procedure setBigMarkLength(value: longint);
    function getSmallMarkWidth:longint;
    procedure setSmallMarkWidth(value: longint);
    function getBigMarkWidth:longint;
    procedure setBigMarkWidth(value: longint);

  protected
    { Protected-Deklarationen }
    procedure Paint; override;
    procedure ComponentResize(Sender: TObject);
  public
    { Public-Deklarationen }
    constructor Create(AOwner:TComponent); override;
    destructor destroy; override;
  published
    { Published-Deklarationen }
    property Max:extended read getMax write setMax;
    property Min:extended read getMin write setMin;
    property Position:extended read getPosition write setPosition;
    property CaptionDistance:extended read getCaptionDistance write setCaptionDistance;
    property BigMarkDistance:extended read getBigMarkDistance write setBigMarkDistance;
    property SmallMarkDistance:extended read getSmallMarkDistance write setSmallMarkDistance;
    property ShowUnitName: boolean read getShowUnitName write setShowUnitName;
    property UnitName: string read getUnitName write setUnitName;
    property CaptionFont:TFont read getCaptionFont write setCaptionFont;
    property UnitFont:TFont read getUnitFont write setUnitFont;
    property PositiveColor:TColor read getPositiveColor write setPositiveColor;
    property NegativeColor:TColor read getNegativeColor write detNegativeColor;
    property ZeroColor:TColor read getZeroColor write setZeroColor;
    property IndicatorColor:TColor read getIndicatorColor write setIndicatorColor;
    property BorderColor:TColor read getBorderColor write setBorderColor;
    property BackgroundColor:TColor read getBackgroundColor write setBackgroundColor;
    property IndicatorWidth:Longint read getIndicatorWidth write setIndicatorWidth;
    property BorderWidth:longint read getBorderWidth write setBorderWidth;
    property SmallMarkLength:longint read getSmallMarkLength write setSmallMarkLength;
    property BigMarkLength:longint read getBigMarkLength write setBigMarkLength;
    property SmallMarkWidth:longint read getSmallMarkWidth write setSmallMarkWidth;
    property BigMarkWidth:longint read getBigMarkWidth write setBigMarkWidth;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('JKSoft', [TJKMeter]);
end;


constructor TJKMeter.Create(AOwner:TComponent);
begin
  inherited create(AOwner);

  FTempBitMap:=TBitMap.create;
  OnResize:=ComponentResize;
  width:=200;
  height:=round(width*5/8);
  FCaptionFont:=TFont.create;
  FCaptionFont.Name:='Arial';
  FCaptionFont.height:=-10;
  FUnitFont:=TFont.create;
  FUnitFont.Name:='Arial';
  FUnitFont.height:=-15;
  FUnitFont.Style:=[fsbold];
  FUnitName:='[Einheit]';
  FShowUnitName:=true;
  FPosColor:=clBlack;
  FNegColor:=clRed;
  FZeroColor:=clGreen;
  FIndicatorColor:=clBlack;
  FBorderColor:=clBlack;
  FBackgroundColor:=clWhite;
  FIndicatorwidth:=4;
  FBorderWidth:=2;
  FMax:=10;
  FMin:=-10;
  FPosition:=0;
  FCaptionDistance:=2;
  FBigMarkDistance:=1;
  FSmallMarkDistance:=0.5;
  FSmallMarkLength:=3;
  FBigMarkLength:=6;
  FSmallMarkWidth:=1;
  FBigMarkWidth:=2;
end;

destructor TJKMeter.destroy;
begin
  FCaptionFont.free;
  FUnitFont.free;
  FTempBitMap.free;

  inherited destroy;
end;

procedure TJKMeter.PaintScale;
var radius, i, mittelX, mittelY: longint;
    winkel, Dmax, textDRad: real;
    col: TColor;
    bigMarkNum, smallMarkNum, captionNum:longint;
    text:string;
begin
  FTempBitMap.Canvas.font.Assign(FCaptionFont);
  DMax:=abs(FMax-FMin);
  bigMarkNum:= round(DMax / FBigMarkDistance);
  smallMarkNum:= round(DMax / FSmallMarkDistance);
  captionNum:= round(DMax / FCaptionDistance);
  mittelX:=width div 2;
  mittelY:= height-5-FIndicatorWidth;
  radius:= 8* round(height/10)-1-FIndicatorWidth;
  FTempBitMap.Canvas.Pixels[mittelx,mittely]:=clBlack;
  textDRad:=0.75*sqrt(sqr(FTempBitMap.Canvas.TextHeight('1'))+sqr(FTempBitMap.Canvas.Textwidth('000')));

  if smallMarkNum>0 then begin
    winkel:=(pi/2)/smallMarkNum;
    FTempBitMap.Canvas.pen.Width:=FSmallMarkWidth;
    for i:=0 to smallMarkNum do begin
      col:=FNegColor;
      if (FMax-(i/smallMarkNum*DMax))>0 then col:=FPosColor;
      if (FMax-(i/smallMarkNum*DMax))=0 then col:=FZeroColor;
      FTempBitMap.Canvas.pen.color:=col;
      FTempBitMap.Canvas.moveto(mittelX-Round(radius*sin((i*winkel)-(pi/4))), mittelY-Round(radius*Cos((i*winkel)-(pi/4))));
      FTempBitMap.Canvas.lineto(mittelX-Round((radius+FSmallMarkLength)*sin((i*winkel)-(pi/4))), mittelY-Round((radius+FSmallMarkLength)*Cos((i*winkel)-(pi/4))));
    end;
  end;

  if bigMarkNum>0 then begin
    winkel:=(pi/2)/bigMarkNum;
    FTempBitMap.Canvas.pen.Width:=FBigMarkWidth;
    for i:=0 to bigMarkNum do begin
      col:=FNegColor;
      if (FMax-(i/bigMarkNum*DMax))>0 then col:=FPosColor;
      if (FMax-(i/bigMarkNum*DMax))=0 then col:=FZeroColor;
      FTempBitMap.Canvas.pen.color:=col;
      FTempBitMap.Canvas.moveto(mittelX-Round(radius*sin((i*winkel)-(pi/4))), mittelY-Round(radius*Cos((i*winkel)-(pi/4))));
      FTempBitMap.Canvas.lineto(mittelX-Round((radius+FBigMarkLength)*sin((i*winkel)-(pi/4))), mittelY-Round((radius+FBigMarkLength)*Cos((i*winkel)-(pi/4))));
    end;
  end;

  if CaptionNum>0 then begin
    winkel:=(pi/2)/captionNum;
    for i:=0 to CaptionNum do begin
      col:=FNegColor;
      if (FMax-(i/CaptionNum*DMax))>0 then col:=FPosColor;
      if (FMax-(i/CaptionNum*DMax))=0 then col:=FZeroColor;
      FTempBitMap.Canvas.font.color:=col;
      text:=floattostr(FMax-(i/CaptionNum*DMax));
      FTempBitMap.Canvas.textout(mittelX-(FTempBitMap.Canvas.textwidth(text) div 2)-Round((radius+FBigMarkLength+textDRad)*sin((i*winkel)-(pi/4))), mittelY-Round((radius+FBigMarkLength+textDRad)*Cos((i*winkel)-(pi/4))), text);
    end;
  end;
end;

procedure TJKMeter.PaintIndicator;
var radius, mittelX, mittelY: longint;
    winkel, Dmax, pos: real;
begin
  DMax:=abs(FMax-FMin);
  mittelX:=width div 2;
  mittelY:= height-5-FIndicatorWidth;
  radius:= 8* round(height/10);

  pos:=FPosition;
  if FPosition>FMax then pos:=FMax;
  if FPosition<FMin then pos:=FMin;

  winkel:=(abs(Pos-FMin)/DMax*(pi/2))-(pi/4);
  FTempBitMap.Canvas.pen.Width:=FIndicatorWidth;
  FTempBitMap.Canvas.Pen.Color:=FIndicatorColor;
  FTempBitMap.Canvas.moveto(mittelX, mittelY);
  FTempBitMap.Canvas.lineto(mittelX+Round(radius*sin(winkel)), mittelY-Round(radius*Cos(winkel)));
  FTempBitMap.Canvas.Pixels[mittelx,mittely]:=clBlack;
end;

procedure TJKMeter.PaintHousing;
begin
  FTempBitMap.Canvas.pen.Style:=psSolid;
  FTempBitMap.Canvas.pen.color:=clBlack;
  FTempBitMap.Canvas.pen.mode:=pmCopy;
  FTempBitMap.Canvas.brush.Style:=bsSolid;
  FTempBitMap.Canvas.brush.color:=clBlack;
  FTempBitMap.Canvas.Rectangle(0,0,width,height);
  FTempBitMap.Canvas.brush.color:=FBackGroundColor;
  FTempBitMap.Canvas.pen.color:=FBorderColor;
  FTempBitMap.Canvas.Pen.width:=FBorderWidth;
  FTempBitMap.Canvas.Rectangle(1,1,width,height);
  FTempBitMap.Canvas.Font.assign(FUnitFont);
  FTempBitMap.Canvas.TextOut(FBorderwidth+5,height-FBorderwidth-5-FTempBitMap.Canvas.textheight(FUnitName),FUnitName);
end;

procedure TJKMeter.ClearCanvas;
var rect:TRect;
begin
  FTempBitMap.Canvas.brush.style:=bsSolid;
  FTempBitMap.Canvas.brush.color:=clWhite;
  FTempBitMap.Canvas.pen.mode:=pmCopy;
  FTempBitMap.Canvas.pen.style:=psSolid;
  FTempBitMap.Canvas.pen.color:=clWhite;
  rect:=FTempBitMap.Canvas.ClipRect;
  FTempBitMap.Canvas.fillrect(rect);
end;

procedure TJKMeter.Paint;
begin
  FTempBitMap.width:=width;
  FTempBitMap.height:=height;
  ClearCanvas;
  PaintHousing;
  PaintScale;
  PaintIndicator;
  Canvas.Draw(0,0,FTempBitMap);
end;

procedure TJKMeter.ComponentResize(Sender: TObject);
begin
  height:=round(width*5/8);
end;

function TJKMeter.getMax:extended;
begin result:=FMax; end;

procedure TJKMeter.setMax(value: extended);
begin FMax:=value; Paint; end;

function TJKMeter.getMin:extended;
begin result:=FMin end;

procedure TJKMeter.setMin(value: extended);
begin FMin:=value; Paint; end;

function TJKMeter.getPosition:extended;
begin result:=FPosition end;

procedure TJKMeter.setPosition(value: extended);
begin Fposition:=value; Paint; end;

function TJKMeter.getCaptionDistance:extended;
begin result:=FCaptionDistance end;

procedure TJKMeter.setCaptionDistance(value: extended);
begin FCaptionDistance:=value; Paint; end;

function TJKMeter.getBigMarkDistance:extended;
begin result:=FBigMarkDistance end;

procedure TJKMeter.setBigMarkDistance(value: extended);
begin FBigMarkDistance:=value; Paint; end;

function TJKMeter.getSmallMarkDistance:extended;
begin result:=FSmallMarkDistance end;

procedure TJKMeter.setSmallMarkDistance(value: extended);
begin FSmallMarkDistance:=value; Paint; end;

function TJKMeter.getShowUnitName:boolean;
begin result:=FShowUnitName end;

procedure TJKMeter.setShowUnitName(value: boolean);
begin FShowUnitName:=value; Paint; end;

function TJKMeter.getUnitName:string;
begin result:=FUnitName end;

procedure TJKMeter.setUnitName(value: string);
begin FUnitName:=value; Paint; end;

function TJKMeter.getCaptionFont:TFont;
begin result:=FCaptionFont end;

procedure TJKMeter.setCaptionFont(value:TFont);
begin FCaptionFont.assign(value); Paint; end;

function TJKMeter.getUnitFont:TFont;
begin result:=FUnitFont end;

procedure TJKMeter.setUnitFont(value:TFont);
begin FUnitFont.assign(value); Paint; end;

function TJKMeter.getPositiveColor:TColor;
begin result:=FPosColor; end;

procedure TJKMeter.setPositiveColor(value: TColor);
begin FPosColor:=value; Paint; end;

function TJKMeter.getNegativeColor:TColor;
begin result:=FNegColor; end;

procedure TJKMeter.detNegativeColor(value: TColor);
begin FNegColor:=value; Paint; end;

function TJKMeter.getZeroColor:TColor;
begin result:=FZeroColor end;

procedure TJKMeter.setZeroColor(value: TColor);
begin FZeroColor:=value; Paint; end;

function TJKMeter.getIndicatorColor:TColor;
begin result:=FIndicatorcolor end;

procedure TJKMeter.setIndicatorColor(value: TColor);
begin FIndicatorColor:=value; Paint; end;

function TJKMeter.getBorderColor:TColor;
begin result:=FBorderColor end;

procedure TJKMeter.setBorderColor(value: TColor);
begin FBorderColor:=value; Paint; end;

function TJKMeter.getBackgroundColor:TColor;
begin result:=FBackGroundColor end;

procedure TJKMeter.setBackgroundColor(value: TColor);
begin FBackgroundColor:=value; Paint; end;

function TJKMeter.getIndicatorWidth:longint;
begin result:=FIndicatorWidth end;

procedure TJKMeter.setIndicatorWidth(value: longint);
begin FIndicatorwidth:=value; Paint; end;

function TJKMeter.getBorderWidth:longint;
begin result:=FBorderWidth end;

procedure TJKMeter.setBorderWidth(value: longint);
begin FBorderWidth:=value; Paint; end;

function TJKMeter.getSmallMarkLength:longint;
begin result:=FSmallMarkLength end;

procedure TJKMeter.setSmallMarkLength(value: longint);
begin FSmallMarkLength:=value; Paint; end;

function TJKMeter.getBigMarkLength:longint;
begin result:=FBigMarkLength end;

procedure TJKMeter.setBigMarkLength(value: longint);
begin FBigMarkLength:=value; Paint; end;

function TJKMeter.getSmallMarkWidth:longint;
begin result:=FSmallMarkwidth end;

procedure TJKMeter.setSmallMarkWidth(value: longint);
begin FSmallMarkWidth:=value; Paint; end;

function TJKMeter.getBigMarkWidth:longint;
begin result:=FBigMarkwidth end;

procedure TJKMeter.setBigMarkWidth(value: longint);
begin FBigMarkWidth:=value; Paint; end;


end.
