unit jkErrorCalculation;
{
  this unit provides datatypes and functions to do calculation with values
  that contain an error

  (c) 2003 by Jan W. Krieger
  www.jkrieger.de -- jan@jkrieger.de
}

interface

uses math, sysutils;

const ErrorSeparator='+/-';

type
  // an ErrorDouble-record always contains the value itself an the absolute error
  //                                                    (not the relative error!)
  ErrorDouble=record
    value, error:extended;
  end;

function Add(x, y:ErrorDouble):ErrorDouble; overload;
function Add(x:ErrorDouble; y:double):ErrorDouble; overload;
function Subtract(x, y:ErrorDouble):ErrorDouble; overload;
function Subtract(x:ErrorDouble; y:double):ErrorDouble; overload;
function Subtract(x:double; y:ErrorDouble):ErrorDouble; overload;
function Mul(x, y: ErrorDouble): ErrorDouble; overload;
function Mul(x:double; y: ErrorDouble): ErrorDouble; overload;
function Divide(x, y: ErrorDouble): ErrorDouble; overload;
function Divide(x:double; y: ErrorDouble): ErrorDouble; overload;
function Divide(x: ErrorDouble; y:double): ErrorDouble; overload;
function RelativeError(x: ErrorDouble): double;
function Power(base: ErrorDouble; exponent: double): ErrorDouble; overload;
function Power(base, exponent: ErrorDouble): ErrorDouble; overload;
function Power(base, exponent: extended): extended; overload;
function Exp(x: extended): extended; overload;
function Exp(x: ErrorDouble): ErrorDouble; overload;
function Ln(x: ErrorDouble): ErrorDouble; overload;
function Ln(x: extended): extended; overload;
function LogN(N, x: extended): extended; overload;
function LogN(N:extended; x: ErrorDouble): ErrorDouble; overload;
function ErrorD(x, DeltaX: double): ErrorDouble;
function FloatToStr(x: ErrorDouble): string; overload;
function FloatToStr(Value: Extended): string; overload;
function sin(x: extended): extended; overload;
function sin(x: ErrorDouble): ErrorDouble; overload;
function cos(x: extended): extended; overload;
function cos(x: ErrorDouble): ErrorDouble; overload;
function tan(x: extended): extended; overload;
function tan(x: ErrorDouble): ErrorDouble; overload;
function cot(x: extended): extended; overload;
function cot(x: ErrorDouble): ErrorDouble; overload;
function arcsin(x: extended): extended; overload;
function arcsin(x: ErrorDouble): ErrorDouble; overload;
function arccos(x: extended): extended; overload;
function arccos(x: ErrorDouble): ErrorDouble; overload;
function arctan(x: extended): extended; overload;
function arctan(x: ErrorDouble): ErrorDouble; overload;
function sqrt(x: extended): extended; overload;
function sqrt(x: ErrorDouble): ErrorDouble; overload;
function Max(x,y:ErrorDouble):ErrorDouble; overload
function Min(x,y:ErrorDouble):ErrorDouble; overload
function hypot(x,y:extended):extended; overload
function hypot(x,y:ErrorDouble):ErrorDouble; overload
function sqr(x: ErrorDouble): ErrorDouble; overload;
function sqr(x: extended): extended; overload;
function sinh(x: extended): extended; overload;
function sinh(x: ErrorDouble): ErrorDouble; overload;
function cosh(x: extended): extended; overload;
function cosh(x: ErrorDouble): ErrorDouble; overload;
function tanh(x: extended): extended; overload;
function tanh(x: ErrorDouble): ErrorDouble; overload;
function arcsinh(x: extended): extended; overload;
function arcsinh(x: ErrorDouble): ErrorDouble; overload;
function arccosh(x: extended): extended; overload;
function arccosh(x: ErrorDouble): ErrorDouble; overload;
function arctanh(x: extended): extended; overload;
function arctanh(x: ErrorDouble): ErrorDouble; overload;

implementation

//add two values with errors
function Add(x, y:ErrorDouble):ErrorDouble;
begin
  result.value:=x.value+y.value;
  result.error:=sqrt(x.error*x.error+y.error*y.error);
end;

function Add(x:ErrorDouble; y:double):ErrorDouble;
begin
  result.value:=x.value+y;
  result.error:=x.error;
end;

function Subtract(x, y:ErrorDouble):ErrorDouble;
begin
  result.value:=x.value-y.value;
  result.error:=sqrt(x.error*x.error+y.error*y.error);
end;

function Subtract(x:ErrorDouble; y:double):ErrorDouble;
begin
  result.value:=x.value-y;
  result.error:=x.error;
end;

function Subtract(x:double; y:ErrorDouble):ErrorDouble;
begin
  result.value:=x-y.value;
  result.error:=y.error;
end;

//multiply two values with errors
function Mul(x, y: ErrorDouble): ErrorDouble;
begin
  result.value:=x.value*y.value;
  result.error:=sqrt(sqr(y.value*x.error)+sqr(x.value*y.error));
end;

//multiply a value with error with a value without an error
function Mul(x:double; y: ErrorDouble): ErrorDouble;
begin
  result.value:=x*y.value;
  result.error:=abs(x)*y.error;
end;

// x/y with errors
function Divide(x, y: ErrorDouble): ErrorDouble;
begin
  result.value:=x.value/y.value;
  result.error:=sqrt(sqr(x.error/y.value)+sqr(x.value*y.error/sqr(y.value)));
end;

function Divide(x:double; y: ErrorDouble): ErrorDouble;
begin
  result.value:=x/y.value;
  result.error:=abs(y.error*x/sqr(y.value));
end;

function Divide(x: ErrorDouble; y:double): ErrorDouble;
begin
  result:=Mul(1/y, x);
end;

function RelativeError(x: ErrorDouble): double;
begin
  result:=x.error/x.value;
end;

function Power(base: ErrorDouble; exponent: double): ErrorDouble;
begin
  result.value:=math.Power(base.value, exponent);
  result.error:=abs(math.power(base.value, exponent-1)*base.error*exponent);
end;

//replaces the power-function from math.pas (allows to overload it)
function Power(base, exponent: extended): extended;
begin
  result:=math.Power(base, exponent);
end;

function Power(base, exponent: ErrorDouble): ErrorDouble;
begin
  if base.value=0 then begin
    result:=ErrorD(0,0);
  end else if exponent.value=0 then begin
    result:=ErrorD(1,0);
  end else begin
    result.value:=math.Power(base.value, exponent.value);
    result.error:=sqrt(sqr(base.error*power(base.value, exponent.value-1)*exponent.value)+
                     sqr(Ln(base.value)*math.power(base.value, exponent.value)*exponent.error));
  end;
end;

function Exp(x: extended): extended;
begin
  result:=system.exp(x);
end;

function Exp(x: ErrorDouble): ErrorDouble;
begin
  result.value:=exp(x.value);
  result.error:=abs(exp(x.value)*x.error);
end;

function Ln(x: ErrorDouble): ErrorDouble;
begin
  result.value:=ln(x.value);
  result.error:=abs(x.error/x.value);
end;

function Ln(x: extended): extended;
begin
  result:=system.Ln(x);
end;


function LogN(N, x: extended): extended;
begin
  result:=math.LogN(n, x);
end;

function LogN(N:extended; x: ErrorDouble): ErrorDouble;
begin
  result.value:=LogN(N, x.value);
  result.error:=abs(x.error/(x.value*Ln(N)));
end;


function ErrorD(x, DeltaX: double): ErrorDouble;
begin
  result.value:=x;
  result.error:=DeltaX;
end;

function FloatToStr(Value: Extended): string;
begin
  result:=sysutils.FloatToStr(Value);
end;

//convert ErrorDouble to string -> output
function FloatToStr(x: ErrorDouble): string;
begin
  if x.error<>0 then result:='('+floattostr(x.value)+' '+ErrorSeparator+floattostr(x.error)+')'
  else result:=floattostr(x.value);
end;

function sin(x: extended): extended;
begin
  result:=system.sin(x);
end;

function sin(x: ErrorDouble): ErrorDouble;
begin
  result.value:=system.sin(x.value);
  result.error:=abs(x.error*system.cos(x.value));
end;

function cos(x:extended):extended;
begin
  result:=system.cos(x);
end;

function cos(x:ErrorDouble):ErrorDouble;
begin
  result.value:=system.cos(x.value);
  result.error:=abs(x.error*system.sin(x.value));
end;

function tan(x:extended):extended;
begin
  result:=math.tan(x);
end;

function tan(x:ErrorDouble):ErrorDouble;
begin
  result.value:=math.tan(x.value);
  result.error:=abs(x.error/sqr(system.cos(x.value)));
end;

function cot(x:extended):extended;
begin
  result:=math.cotan(x);
end;

function cot(x:ErrorDouble):ErrorDouble;
begin
  result.value:=math.cotan(x.value);
  result.error:=abs(x.error/sqr(system.sin(x.value)));
end;

function arcsin(x: extended): extended;
begin
  result:=math.arcsin(x);
end;

function arcsin(x: ErrorDouble): ErrorDouble;
begin
  result.value:=math.arcsin(x.value);
  result.error:=abs(x.error/sqrt(1-sqr(x.value)));
end;

function arccos(x:extended):extended;
begin
  result:=math.arccos(x);
end;

function arccos(x:ErrorDouble):ErrorDouble;
begin
  result.value:=math.arccos(x.value);
  result.error:=abs(x.error/sqrt(1-sqr(x.value)));
end;

function arctan(x:extended):extended;
begin
  result:=system.arctan(x);
end;

function arctan(x:ErrorDouble):ErrorDouble;
begin
  result.value:=system.arctan(x.value);
  result.error:=abs(x.error/(1+sqr(x.value)));
end;

function Max(x,y:ErrorDouble):ErrorDouble; overload
begin
  result:=x;
  if y.value>x.value then result:=y;
end;

function Min(x,y:ErrorDouble):ErrorDouble; overload
begin
  result:=x;
  if y.value<x.value then result:=y;
end;

function sqrt(x:extended):extended;
begin
  result:=system.sqrt(x);
end;

function sqrt(x:ErrorDouble):ErrorDouble;
begin
  result.value:=system.sqrt(x.value);
  result.error:=abs(x.error/(2*system.sqrt(x.value)));
end;

function hypot(x,y:extended):extended;
begin
  result:=math.hypot(x,y);
end;

function hypot(x,y:ErrorDouble):ErrorDouble;
{ aus math.h bernommen !!!

  formula: Sqrt(X*X + Y*Y)
  implemented as:  |Y|*Sqrt(1+Sqr(X/Y)), |X| < |Y| for greater precision}
var
  Temp: ErrorDouble;
begin
  X.value := Abs(X.value);
  Y.value := Abs(Y.value);
  if X.value > Y.value then
  begin
    Temp := X;
    X := Y;
    Y := Temp;
  end;
  if X.value = 0 then
    Result := Y
  else         // Y > X, X <> 0, so Y > 0
    Result := Mul(Y, Sqrt(Add(Sqr(Divide(X,Y)),1)));
end;

function sqr(x:extended):extended;
begin
  result:=system.sqr(x);
end;

function sqr(x:ErrorDouble):ErrorDouble;
begin
  result:=Mul(x,x);
end;

function sinh(x: extended): extended;
begin
  result:=math.sinh(x);
end;

function sinh(x: ErrorDouble): ErrorDouble;
begin
  result.value:=math.sinh(x.value);
  result.error:=abs(x.error*math.cosh(x.value));
end;

function cosh(x:extended):extended;
begin
  result:=math.cosh(x);
end;

function cosh(x:ErrorDouble):ErrorDouble;
begin
  result.value:=math.cosh(x.value);
  result.error:=abs(x.error*math.sinh(x.value));
end;

function tanh(x:extended):extended;
begin
  result:=math.tanh(x);
end;

function tanh(x:ErrorDouble):ErrorDouble;
begin
  result.value:=math.tanh(x.value);
  result.error:=abs(x.error/sqr(math.cosh(x.value)));
end;

function arcsinh(x: extended): extended;
begin
  result:=math.arcsinh(x);
end;

function arcsinh(x: ErrorDouble): ErrorDouble;
begin
  result.value:=math.arcsinh(x.value);
  result.error:=abs(x.error/sqrt(1+sqr(x.value)));
end;

function arccosh(x:extended):extended;
begin
  result:=math.arccosh(x);
end;

function arccosh(x:ErrorDouble):ErrorDouble;
begin
  result.value:=math.arccosh(x.value);
  result.error:=abs(x.error/sqrt(sqr(x.value)-1));
end;

function arctanh(x:extended):extended;
begin
  result:=math.arctanh(x);
end;

function arctanh(x:ErrorDouble):ErrorDouble;
begin
  result.value:=math.arctanh(x.value);
  result.error:=abs(x.error/(1-sqr(x.value)));
end;



{
function (x:extended):extended;
begin
  result:=(x);
end;

function (x:ErrorDouble):ErrorDouble;
begin
  result.value:=(x.value);
  result.error:=abs(x.error*);
end;

}

end.
