unit jkMathParser;

(*
  TjkMathParser - Formelparser-Komponente fr Delphi

    (c) 2003 by Jan Krieger, Heidelberg, jan@jkrieger.de, www.jkrieger.de
    Dieser Sourcecode ist FREEWARE, dieser Hinweis auf den Autor darf nicht entfernt
    werden, sonst darf jeder diese Quellcode benutzen oder weiterentwickeln, fr jede,
    auch kommerzielle Anwendung. Dieser Quelltext wird ohne jegliche Garantien
    weitergegeben. Einzig das Verkaufen, oder gebhrenpflichtige Abgeben dieses
    Quelltextes, oder Weiterentwicklungen davon ist untersagt (COMPILIERTE PROGRAMME
    drfen verkauft werden !).

    Ich wrde mich ber eine Erwhnung in programmen, die diesen Quelltext benutzen
    freuen. Falls jemand fehler findet, oder substanzielle nderungen/Erweiterungen/
    Verbesserungen vornimmt, so wrde ich mich ber den neuen Quelltext freuen.

  Dieser Funktionsinterpreter basiert auf einem beispiel in dem Buch:
    Bjarne Stroustrup, "Die C++-Programmiersprache"; Addison-Weseley, 2000, S. 116ff

  Folgende Grammatik wird untersttzt:

  program:
     END                                   // END ist Eingabeende
     expression_list


  expression_list:                         // es knne mehrere Ausdrcke, getrennt durch
                                           //   PRINT=';' eingegeben werden. Ein zeilenumbruch
                                           //   wirkt ebenfalls wie PRINT. Der Rckgabewert ent-
                                           //   spricht dem letzten ausgewerteten Ausdruck
     expression [PRINT]                    // das letzte PRINT ist optional (wird von der
                                           //   Komponente angefgt!)
     expression PRINT expression_list


  expression:
     expression + term                     // Addition
     expression - term                     // Subtraktion
     term


  term:
     term / primary                        // Division
     term * primary                        // Multiplikation
     term % primary                        // Modulo (nur fr Ganzzahlwerte, Fehler=0 !!!)
     primary


  primary:
     NUMBER                                // ein Zahl (Formate: 5.3  -5.3   5.3e-4  5.3e4,
                                           //   e/E trennt den Exponenten ab)
                                           //   '.' ist Dezimaltrennzeichen, ',' hat eine andere Bedeutung !!!
     NAME                                  // eine Konstante/Variable
     NAME = expression                     // Zuweisung eines Wertes an eine Konstante/Variable
                                           //   Bsp: e = exp(1);
     - primary
     ( expression )
     | expression |                        // Absolut-Betrag
     NAME( expression )                    // Funktion mit einem Parameter, Bsp: exp(1);
     NAME( expression, expression )        // Funktion mit zwei Parameter, Bsp: logN(10, 1000);
     primary ^ primary                     // Potenz (0^x=0;   x^0=1)
     primary !                             // Fakultt (nur fr Ganzzahlwerte aus [0..25], Fehler=0 !!!)
     {X, deltaX}                           // Wert mit Fehler



  folgende mathematischen Konstanten sind vordefiniert:
     pi,
     e,
     Random/Rnd          // Zufallszahl aus [0, 1]

  folgende physikalischen Konstanten sind vordefiniert:
     m_alpha :=6.6446616e-27;      // [kg] Ruhemasse eines He2+
     m_alpha_u :=4.0015065;        // [u]     -- '' --
     at_unit :=1.660540e-27;       // [kg] 1/12 der Masse des 12C (=1 u)
     avogadro :=6.0221e23;         // [1/mol] Avogrado-Zahl
     boltzmann :=1.3807e-23;       // [J/K] Boltzmann-Konstante
     boltzmann_ev :=8.617386e-5;   // [eV/K]  -- '' --
     efield :=8.8542e-12;          // [C/V m] elektrische Feldkonstante
     m_e :=9.1094e-31;             // [kg] Ruhemasse: Elektron
     m_e_u :=5.48580e-4;           // [u]     -- '' --
     elem_charge :=1.6022e-19;     // [C] Elementarladung
     g_norm :=9.80665;             // [m/s] Fallbeschleunigung, Norm
     g_hd :=9.8101;                // [m/s] Fallbeschleunigung, Heidelberg
     faraday :=96485.309;          // [C/mol] Faraday-Konstante
     gasconst :=8.3145;            // [J/K mol] allg. Gaskonstante
     gravity :=6.673e-11;          // [m/kg s] Gravitationskonstante
     c_light :=2.99792458e8;       // [m/s] Lichtgeschwindigkeit im Vakuum
     m_neutron :=1.67493e-27;      // [kg] Ruhemasse des Neutrons
     m_neutron_u :=1.008665;       // [u]     -- '' --
     v_norm :=22.4;                // [l/mol] molares Vol. eines idealen Gases im Normzustand (1013 mbar, 273 K)
     planck :=6.6261e-34;          // [J s] Planck-Konstante
     planck_ev :=4.1357e-15;       // [eV s]  -- '' --
     m_proton :=1.67262e-27;       // [kg] Ruhemasse: Proton
     m_proton_u :=1.007276;        // [u]     -- '' --
     stefan_boltzmann :=5.6705e-8; // [W/mK^4] Stefan-Boltzmann-Konstante
     m_earth :=5.977e24;           // [kg] Erdmasse
     r_earth :=6268e3;             // [m] Erdradius
     m_sun :=1.98e30;              // [kg] Sonnenmasse
     r_sun :=6.96e8;               // [m] Sonnenradius
     m_moon :=1.23e-2*5.977e24;    // [kg] Mondmasse
     r_moon :=0.273*6268e3;        // [m] Mondradius
     density_h2o :=0.998;          // [kg/l] Dichte von Wasser
     density_air :=1.293;          // [kg/m] Dichte von Luft

  zustzliche Konstanten/Variablen knnen extern verwaltet werden. Dazu mssen den
  zwei Ereignissen OnSetConstant und OnGetConstant Behandlungsroutinen zugewiesen
  werden.
  ber das Ereignis OnGetConstant erfragt dann die Komponente einen Wert fr eine
  Konstante (Name: <name>). Der ermittelte Wert wird in <value> zurckgegeben. Wurde
  ein Wert zurckgegeben, so muss auch der Parameter <ConstSet> auf true gesetzt werden.
  Dies zeigt der Komponente, dass die Konstante dem Programm bekannt war. Wird intern
  zu einem Namen keine Konstante gefunden und ist <ConstSet> nach einem Aufruf von
  OnGetConstant false, dann wird eine meldung ausgegeben, dass die Konstante nicht
  gefunden wurde (auer es folgt ein '=', dann wird die Konstante intern angelegt
  und ihr ein Wert zugewiesen.
  ber das Ereignis OnSetConstant kann das ausfhrende Programm auf das setzen einer
  externen Konstante reagieren. Wurde die Anfrage (erfolgreich) bearbeitet, so muss,
  wie oben, <ConstSet> auf true gesetzt werden.
  Diese beiden Ereignisse erlauben es nicht nur externe Konstanten zu verwenden, sondern
  auch externe Variablen. Ein Beispiel:

      var x:int64
      procedure GetC(Sender: TjkMathParser; name:string; var value:extended; var constSet:boolean);
      begin
        if name='x' then begin // wenn nach Variable 'x' gefragt wird, dann Wert zurckgeben
          value:=x;
          constSet:=true;
        end;
      end;
      procedure SetC(Sender: TjkMathParser; name:string; value:extended; var constSet:boolean);
      begin
        if name='x' then begin // wenn nach Variable 'x' gefragt wird, dann Wert setzen
          x:=trunc(value);
          constSet:=true;
        end;
      end;


  folgende einparametrigen Funktionen sind vordefiniert:
     sqr                        // Quadrat des Argumentes
     sqrt                       // Quadratwurzel
     exp                        // Exponentialfunktion
     ln                         // natrlicher Logarithmus
     log10, log, lg             // Logarithmus zur Basis 10
     log2, lb                   // Logarithmus zur Basis 2
     abs                        // Absolutbetrag (alternativ zu |...|)
     random, rnd                // Zufallszahl aus [0, Argument]
     randomint, rndint          // ganzzahlige Zufallszahl aus [0, Argument]
     arccos, arcsin, arctan     // Arcus-Funktionen (trigonometrische Umkehrfkt.)
     arccosh, arcsinh, arctanh  // hyperbol. Arcus-Funktionen
     cos, sin, tan, cot         // trigonometrische Funktionen
     cosh, sinh, tanh           // hyperbolische Funktionen
     ceil                       // kleinster Ganzzahl-Wert, der grer oder gleich X
     floor                      // Abrunden
     round                      // Runden
     frac                       // Nachkommaanteil
     int                        // ganzzahliger Anteil
     sgn                        // Signums-/Vorzeichen-Funktion
     error, relativeerror       // gibt den Fehler des Parameters zurck
     relerror                   // fibt den relativen fehler des Parameters zurck

  folgende zweiparametrigen Funktionen sind vordefiniert:
     logn(N, ex)                   // Logarithmus zu Basis <N> von <ex>
     randg, randomg(Mean, StdDev)  // RandG generiert Zufallszahlen anhand der Gauschen
                                   // Normalverteilung um das arithmetische Mittel (Mean)
     hypot(X, Y)                   // Hypotenuse eines rechtwinkligen Dreiecks (Sqrt(X**2 + Y**2))
     min                           // gibt das Minimum von zwei Werten zurck
     max                           // gibt das Maximum von zwei Werten zurck



Revision History:
  2.Mar.2003:
    Schreiben der Grundfunktionen und Test

  3.Mar.2003:
    - TjkMap-Klasse implementiert
    - Verhalten von Potenz korrigiert
    - Hex-darstellung von Zahlen
    - zustzliche externe (ereignisgesteuerte) Konstantenverwaltung
    - Integer-Arithmetik (Module, Fakultt)
    - Fehlerrechnung


TODO:
*)

interface

uses
  sysutils,
  classes,
  jkClasses,
  math,
  JKErrorCalculation;

type
  // berschreibt die definition aus JKErrorCalculation, um sie ohne diese unit nutzbar
  // zu machen
  ErrorDouble=JKErrorCalculation.ErrorDouble;

const
  jkpWhiteSpaces=[' ', '	', #0];
type
  TjkMathParser=class; //forward-Deklaration

  tjkpTokenType=(jkptNone, jkpTName, jkpTNumber, jkpTPlus, jkpTMinus, jkpTMul,
                 jkpTPrint, jkpTDiv, jkptEnd, jkptPower, jkptLB, jkptRB, jkptMOD,
                 jkptABSBrackets, jkptLEB, jkptREB, jkptASSIGN, jkptFUNCTION,
                 jkptFactorial, jkptARGUMENTDIVIDER);

  tjkpToken=record
    typ:tjkpTokenType;
    value:ErrorDouble;
    name:string;
  end;

  tjkpSingleParameterFunction=function(param:ErrorDouble; var error:string):ErrorDouble;
  tjkpDoubleParameterFunction=function(param1, param2:ErrorDouble; var error:string):ErrorDouble;
  // Funktionen, die registriert werden sollen mssen diesen Typ haben.
  // param* enthllt die Parameter, in error kann eine Fehlermeldung zurck-
  // gegeben werden. falls kein Fehler auftritt einfach auf error='' belassen
  tjkpFunctionType=(jkpftSingle, jkpftDouble, jkpftNone);
  pjkpFunction=^tjkpFunction;
  tjkpFunction=record
    singleFunction: tjkpSingleParameterFunction;
    doubleFunction: tjkpDoubleParameterFunction;
    typ: tjkpFunctionType;
    name:string[255];
  end;
  tjkpGetConstant=procedure(Sender: TjkMathParser; name:string; var value:ErrorDouble; var constSet:boolean);
  tjkpSetConstant=procedure(Sender: TjkMathParser; name:string; value:ErrorDouble; var constSet:boolean);

  TjkMathParser=class
  private
    FFormula:string;
    FPosition, FLength:longint;
    FCurrentToken:tjkpToken;
    FNumberOfErrors:longint;
    Fconstants:TjkMap;
    FFunctionList:TList;
    FNoFunctions:tjkpFunction;
    FExternalGetConst:tjkpGetConstant;
    FExternalSetConst:tjkpSetConstant;
    function getFunction(fname: string): tjkpFunction;
    function IsInteger(value: extended): boolean;
    
  protected
    function getNAME(text: string; pos: Integer; var name: string): longint;
    function getToken: tjkpToken;
    procedure Error(text: string);
    function FormatString(text: string): string;
    function Expression(getTok: boolean): ErrorDouble;
    function Term(getTok: boolean): ErrorDouble;
    function Primary(getTok: boolean): ErrorDouble;
    function getNumber(text: string; position: Integer; var value: extended): longint;
    function getHexNumber(text: string; position: Integer; var value: extended): longint;
    function getErrorNumber: ErrorDouble;
    function GetConstant(cname: string): ErrorDouble;
  public
    function Parse(formula: string):ErrorDouble;
    constructor create;
    destructor destroy; override;
    procedure AddSingleFunction(fname: string; func: tjkpSingleParameterFunction);
    procedure AddDoubleFunction(fname: string; func: tjkpDoubleParameterFunction);
    procedure AddConstant(name: string; value: ErrorDouble); overload;
    procedure AddConstant(name: string; value: extended); overload;
    procedure DeleteConstant(name: string);
    procedure DeleteFunction(fname: string);
    procedure SetConstant(name: string; value: ErrorDouble);
    property OnGetConstant:tjkpGetConstant read FExternalGetConst write FExternalGetConst;
    property OnSetConstant:tjkpSetConstant read FExternalSetConst write FExternalSetConst;
  end;

function Factorial(value: int64): int64;

implementation

//------------------------------------------------------------------------------
// Standard-Funktionen:
function _abs(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result.value:=abs(param.value);
end;

function _sgn(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result.error:=0;
  if param.value>0 then result.value:=1
    else if param.value=0 then result.value:=0
      else result.value:=-1;
end;

function _ceil(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=ErrorD(ceil(param.value), ceil(param.error));
end;

function _int(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=ErrorD(int(param.value), int(param.error));
end;

function _floor(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=ErrorD(floor(param.value), floor(param.error));
end;

function _round(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=ErrorD(round(param.value), round(param.error));
end;

function _frac(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=ErrorD(frac(param.value), frac(param.error));
end;

function _exp(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=exp(param);
end;

function _sqr(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=Mul(param, param);
end;

function _sqrt(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=ErrorD(0,0);
  if param.value>=0 then result:=sqrt(param)
  else error:='Wurzel von '+floattostr(param)+' nicht definiert';
end;

function _sin(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=sin(param);
end;

function _cos(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=cos(param);
end;

function _sinh(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=sinh(param);
end;

function _cosh(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=cosh(param);
end;

function _tan(param: ErrorDouble; var error:string): ErrorDouble;
begin
  try
    result:=tan(param);
  except
    result:=ErrorD(0,0);
    error:='tan('+floattostr(param)+') nicht definiert';
  end;
end;

function _tanh(param: ErrorDouble; var error:string): ErrorDouble;
begin
  try
    result:=tanh(param);
  except
    result:=ErrorD(0,0);;
    error:='tanh('+floattostr(param)+') nicht definiert';
  end;
end;

function _cot(param: ErrorDouble; var error:string): ErrorDouble;
begin
  try
    result:=cot(param);
  except
    result:=ErrorD(0,0);
    error:='cot('+floattostr(param)+') nicht definiert';
  end;
end;

function _arcsin(param: ErrorDouble; var error:string): ErrorDouble;
begin
  if (param.value<-1)or(param.value>1) then begin
    result:=ErrorD(0,0);
    error:='arcsin() nur fr [-1..1] definiert';
  end else result:=arcsin(param);
end;

function _arctan(param: ErrorDouble; var error:string): ErrorDouble;
begin
  try
    result:=arctan(param);
  except
    result:=ErrorD(0,0);
    error:='arctan('+floattostr(param)+') nicht definiert';
  end;
end;

function _arctanh(param: ErrorDouble; var error:string): ErrorDouble;
begin
  try
    result:=arctanh(param);
  except
    result:=ErrorD(0,0);
    error:='arctanh('+floattostr(param)+') nicht definiert';
  end;
end;

function _arccos(param: ErrorDouble; var error:string): ErrorDouble;
begin
  if (param.value<-1)or(param.value>1) then begin
    result:=ErrorD(0,0);
    error:='arccos() nur fr [-1..1] definiert';
  end else result:=arccos(param);
end;

function _arcsinh(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=arcsinh(param);
end;

function _arccosh(param: ErrorDouble; var error:string): ErrorDouble;
begin
  if (param.value<1) then begin
    result:=ErrorD(0,0);
    error:='arccosh(x) nur fr x>=1 definiert';
  end else result:=arccosh(param);
end;

function _ln(param: ErrorDouble; var error:string): ErrorDouble;
begin
  if param.value<=0 then begin
    result:=ErrorD(0,0);
    error:='ln(x) nur fr x>0 definiert';
  end else result:=ln(param);
end;

function _log10(param: ErrorDouble; var error:string): ErrorDouble;
begin
  if param.value<=0 then begin
    result:=ErrorD(0,0);
    error:='log10(x) nur fr x>0 definiert';
  end else result:=logN(10,param);
end;

function _log2(param: ErrorDouble; var error:string): ErrorDouble;
begin
  if param.value<=0 then begin
    result:=ErrorD(0,0);
    error:='log2(x) nur fr x>0 definiert';
  end else result:=logN(2, param);
end;

function _LogN(param1, param2: ErrorDouble; var error:string): ErrorDouble;
begin
  if param2.value<=0 then begin
    result:=ErrorD(0,0);
    error:='log(x) nur fr x>0 definiert';
  end else result:=LogN(param1.value, param2);
end;

function _RandG(param1, param2: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=ErrorD(randG(param1.value, param2.value), 0);
end;

function _Random(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=ErrorD(Random(Trunc(param.value)*1000)/1000, 0);
end;

function _RandomInt(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=ErrorD(Random(Trunc(param.value)), 0);
end;

function _hypot(param1, param2: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=Hypot(param1, param2);
end;

function _Max(param1, param2: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=max(param1, param2);
end;

function _Min(param1, param2: ErrorDouble; var error:string): ErrorDouble;
begin
  result:=Min(param1, param2);
end;

function _error(param: ErrorDouble; var error: string): ErrorDouble;
begin
  result:=ErrorD(param.error, 0);
end;

function _relativeerror(param: ErrorDouble; var error: string): ErrorDouble;
begin
  result:=ErrorD(relativeError(param), 0)
end;


//------------------------------------------------------------------------------
//Leer-Funktionen
function _nothings(param: ErrorDouble; var error:string): ErrorDouble;
begin
  result.value:=0;
  result.error:=0;
end;

function _nothingd(param1,param2: ErrorDouble; var error:string): ErrorDouble;
begin
  result.value:=0;
  result.error:=0;
end;


// TjkMathParser -----------------------------------------------------------------

//registriert eine Funktion mit zwei Parametern
procedure TjkMathParser.AddConstant(name: string; value: ErrorDouble);
var e:^ErrorDouble;
begin
  new(e);
  e^:=value;
  FConstants[name]:=e;
end;

procedure TjkMathParser.AddConstant(name: string; value: extended);
var e:^ErrorDouble;
begin
  new(e);
  e^.value:=value;
  e^.error:=0;
  FConstants[name]:=e;
end;

procedure TjkMathParser.AddDoubleFunction(fname: string; func: tjkpDoubleParameterFunction);
var f:^tjkpFunction;
begin
  new(f);
  f^.name:=fname;
  f^.typ:=jkpftDouble;
  f^.doubleFunction:=func;
  FFunctionList.add(f);
end;

//registriert eine Funktion mit einem Parametern
procedure TjkMathParser.AddSingleFunction(fname: string; func: tjkpSingleParameterFunction);
var f:^tjkpFunction;
begin
  new(f);
  f^.name:=fname;
  f^.typ:=jkpftSingle;
  f^.singleFunction:=func;
  FFunctionList.add(f);
end;

constructor TjkMathParser.create;
begin
  inherited create;
  FFormula:='';
  FNumberOfErrors:=0;
  FPosition:=1;
  FCurrentToken.typ:=jkptNone;
  FCurrentToken.value.value:=0;
  FCurrentToken.value.error:=0;
  FCurrentToken.name:='';
  Fconstants:=TjkMap.create;

  //Standard-Konstanten hinzufgen (Random/Rnd sind eigens in Primary() implementiert, keine Standard-konstanten !!!)
  //mathematisch:
    AddConstant('pi', Pi);
    AddConstant('e', exp(1));

  //physikalisch:
    AddConstant('m_alpha', 6.6446616e-27);      // [kg] Ruhemasse eines He2+
    AddConstant('m_alpha_u', 4.0015065);        // [u]     -- '' --
    AddConstant('at_unit', 1.660540e-27);       // [kg] 1/12 der Masse des 12C (=1 u)
    AddConstant('avogadro', 6.0221e23);         // [1/mol] Avogrado-Zahl
    AddConstant('boltzmann', 1.3807e-23);       // [J/K] Boltzmann-Konstante
    AddConstant('boltzmann_ev', 8.617386e-5);   // [eV/K]  -- '' --
    AddConstant('efield', 8.8542e-12);          // [C/V m] elektrische Feldkonstante
    AddConstant('m_e', 9.1094e-31);             // [kg] Ruhemasse: Elektron
    AddConstant('m_e_u', 5.48580e-4);           // [u]     -- '' --
    AddConstant('elem_charge', 1.6022e-19);     // [C] Elementarladung
    AddConstant('g_norm', 9.80665);             // [m/s] Fallbeschleunigung', Norm
    AddConstant('g_hd', 9.8101);                // [m/s] Fallbeschleunigung', Heidelberg
    AddConstant('faraday', 96485.309);          // [C/mol] Faraday-Konstante
    AddConstant('gasconst', 8.3145);            // [J/K mol] allg. Gaskonstante
    AddConstant('gravity', 6.673e-11);          // [m/kg s] Gravitationskonstante
    AddConstant('c_light', 2.99792458e8);       // [m/s] Lichtgeschwindigkeit im Vakuum
    AddConstant('m_neutron', 1.67493e-27);      // [kg] Ruhemasse des Neutrons
    AddConstant('m_neutron_u', 1.008665);       // [u]     -- '' --
    AddConstant('v_norm', 22.4);                // [l/mol] molares Vol. eines idealen Gases im Normzustand (1013 mbar', 273 K)
    AddConstant('planck', 6.6261e-34);          // [J s] Planck-Konstante
    AddConstant('planck_ev', 4.1357e-15);       // [eV s]  -- '' --
    AddConstant('m_proton', 1.67262e-27);       // [kg] Ruhemasse: Proton
    AddConstant('m_proton_u', 1.007276);        // [u]     -- '' --
    AddConstant('stefan_boltzmann', 5.6705e-8); // [W/mK^4] Stefan-Boltzmann-Konstante
    AddConstant('m_earth', 5.977e24);           // [kg] Erdmasse
    AddConstant('r_earth', 6268e3);             // [m] Erdradius
    AddConstant('m_sun', 1.98e30);              // [kg] Sonnenmasse
    AddConstant('r_sun', 6.96e8);               // [m] Sonnenradius
    AddConstant('m_moon', 1.23e-2*5.977e24);    // [kg] Mondmasse
    AddConstant('r_moon', 0.273*6268e3);        // [m] Mondradius
    AddConstant('density_h2o', 0.998);          // [kg/l] Dichte von Wasser
    AddConstant('density_air', 1.293);          // [kg/m] Dichte von Luft

  FFunctionList:=TList.Create;

  //Standard-Funktionen registrieren
  AddSingleFunction('sqr', _sqr);
  AddSingleFunction('sqrt', _sqrt);
  AddSingleFunction('exp', _exp);
  AddSingleFunction('ln', _ln);
  AddSingleFunction('log10', _log10); //Zehner-Logarithmus: log10, log, lg
  AddSingleFunction('log', _log10);
  AddSingleFunction('lg', _log10);
  AddSingleFunction('log2', _log2); //Zweier-Logarithmus: lb, log2
  AddSingleFunction('lb', _log2);
  AddSingleFunction('abs', _abs);
  AddSingleFunction('random', _random); //Zufallszahl: random, rnd
  AddSingleFunction('rnd', _random);
  AddSingleFunction('randomint', _randomInt); //ganzzahlige Zufallszahl: randomint, rndint
  AddSingleFunction('rndint', _randomInt);
  AddSingleFunction('arccos', _arccos);
  AddSingleFunction('arcsin', _arcsin);
  AddSingleFunction('arctan', _arctan);
  AddSingleFunction('cos', _cos);
  AddSingleFunction('sin', _sin);
  AddSingleFunction('tan', _tan);
  AddSingleFunction('cot', _cot);
  AddSingleFunction('arccosh', _arccosh);
  AddSingleFunction('arcsinh', _arcsinh);
  AddSingleFunction('arctanh', _arctanh);
  AddSingleFunction('cosh', _cosh);
  AddSingleFunction('sinh', _sinh);
  AddSingleFunction('tanh', _tanh);
  AddSingleFunction('ceil', _ceil);
  AddSingleFunction('floor', _floor);
  AddSingleFunction('round', _round);
  AddSingleFunction('frac', _frac);
  AddSingleFunction('int', _int);
  AddSingleFunction('sgn', _sgn);
  AddSingleFunction('error', _error);
  AddSingleFunction('relativeerror', _relativeerror);
  AddSingleFunction('relerror', _relativeerror);

  AddDoubleFunction('logn', _logn);
  AddDoubleFunction('randg', _randg); //Zufallszahl nach Gau-Verteilung: RandG, RandomG
  AddDoubleFunction('randomg', _randg);
  AddDoubleFunction('hypot', _hypot);
  AddDoubleFunction('min', _min);
  AddDoubleFunction('max', _max);

  //leere Funktionen definieren
  FNoFunctions.name:='';
  FNoFunctions.typ:=jkpftNone;
  FNoFunctions.singleFunction:=_nothings;
  FNoFunctions.doubleFunction:=_nothingd;
end;

procedure TjkMathParser.DeleteConstant(name: string);
begin
  FConstants.delete(name);
end;

procedure TjkMathParser.DeleteFunction(fname: string);
var i:longint;
begin
  i:=0;
  while (i<FFunctionList.count) do begin
    if tjkpFunction(FFunctionList[i]^).name=fname then begin
      i:=FFunctionList.count;
      FFunctionList.Delete(i);
    end;
    inc(i);
  end;
end;

destructor TjkMathParser.destroy;
var i:longint;
begin
  FConstants.free; //Konstanten-Speicher freigeben

  //Funkctionen-Liste freigeben
  if FFunctionList.count>0 then for i:=0 to FFunctionList.count-1 do
    try dispose(FFunctionList[i]); except end;
  FFunctionList.Clear;
  FFunctionList.free;
  
  inherited destroy;
end;

// Fehler-Behandlung
procedure TjkMathParser.Error(text: string);
begin
  inc(FNumberOfErrors);
  writeln('Fehler: '+text+'    [Nr. '+inttostr(FNumberOfErrors)+'][Pos. '+inttostr(FPosition)+']');
end;

// Addition und Subtraktion 
function TjkMathParser.Expression(getTok: boolean): ErrorDouble;
var left:ErrorDouble;
    loop:boolean;
begin
  left:=term(getTok);
  loop:=true;
  result.value:=0;
  result.error:=0;
  while loop do begin
    case FCurrentToken.typ of
      jkptPLUS: left:=Add(left, term(true));
      jkptMINUS: left:=Subtract(left, term(true));
    else
      result:= left;
      loop:=false;
    end;
  end;
end;

//Einen String fr das parsen vorbereiten:
//  - ';' am Ende anfgen
//  - alle #13 in #10 umwandeln (-> eindeutiger Zeilenumbruch durch ein Zeichen gekennzeichnet)
//  - alles in Kleinbuchstaben (-> nicht case-sensitive)
function TjkMathParser.FormatString(text: string): string;
begin
  result:=ANSILowerCase(text);
  result:=stringreplace(result, #13, #10, [rfReplaceAll, rfIgnoreCase]);
  result:=stringreplace(result, ' ', '', [rfReplaceAll, rfIgnoreCase]);
  if length(result)>0 then
    if result[length(result)]<>';' then result:=result+';';
end;

// sucht in der internen Funktionenliste nach <fname> und gibt die entsprechenden
// Daten zurck
function TjkMathParser.getFunction(fname: string): tjkpFunction;
var i:longint;
begin
  i:=0;
  result:=FNoFunctions;
  while (i<FFunctionList.count) do begin
    if tjkpFunction(FFunctionList[i]^).name=fname then begin
      result:=tjkpFunction(FFunctionList[i]^);
      i:=FFunctionList.count;
    end;
    inc(i);
  end;
end;

function TjkMathParser.getNAME(text: string; pos: Integer; var name: string): longint;
{
  extrahiert einen Bezeichner aus <text> (Funktionenname, Variablenname),
  sonst wie getNumber.
  In Namen sind erlaubt:
    erstes Zeichen:   ['a'..'z', '', '_']
    folgende Zeichen: ['a'..'z', '', '_', '0'..'9']
}
var isName:boolean;
begin
    result:=pos;
    isName:=true;
    name:='';
    while isName do begin
      if (text[result] in['a'..'z', '', '_']) then name:=name+text[result]
        else if (text[result] in['a'..'z', '', '_', '0'..'9'])and(name<>'') then name:=name+text[result]
          else isName:=false;
      inc(result);
      if result>length(text) then begin isName:=false; inc(result); end;
    end;
    result:=result-2;
end;


// liet aus <text> ab Position <position> eine Zahl ein. Das Ergebnis wird in <value>
// zurckgegeben. Der Rckgabewert der Funktion entspricht der letzten Stelle der eingelesenen
// Zahl im String.
//
// erlaubte Formate:
//    5.3  -5.3   5.3e-4  5.3e4  
function TjkMathParser.getNumber(text: string; position: Integer; var value: extended): longint;
var number:string;
    isNumber, isExponent, isExponentN:boolean;
begin
  result:=position;
  isExponent:=false;
  isExponentN:=false;
  isNumber:=true;
  number:='';
  while isNumber do begin
    if text[result] in['0'..'9', '.'] then number:=number+text[result] else
      if (text[result] in['-', '+'])and(number='') then number:=number+text[result] else
        if (not isExponent)and(text[result] in['e', 'E']) then
          begin number:=number+text[result]; isexponent:=true; end else
          if (isExponent)and(not isExponentN)and(text[result] in['-', '+']) then
            begin number:=number+text[result]; isexponentN:=true; end else isnumber:=false;
    inc(result);
    if result>length(text) then begin isNumber:=false;  inc(result); end;
  end;
  result:=result-2;
  try
    value:=strtofloat(StringReplace(number, '.', ',', [rfReplaceAll, rfIgnoreCase]));
  except
    on e:exception do begin
      value:=0;
      error(e.message);
    end;
  end;
end;

// liet aus <text> ab Position <position> eine Zahl im Hexadecimal-Format ein. Das
// Ergebnis wird in <value> zurckgegeben. Der Rckgabewert der Funktion entspricht
// der letzten Stelle der eingelesenen Zahl im String.
//
// erlaubte Formate:
//    $123cdef  
function TjkMathParser.getHexNumber(text: string; position: Integer; var value: extended): longint;
var number:string;
    isNumber:boolean;
begin
  result:=position;
  isNumber:=true;
  number:='';
  while isNumber do begin
    if text[result] in ['0'..'9', 'a'..'f', '$'] then number:=number+text[result]
      else isnumber:=false;
    inc(result);
    if result>length(text) then begin isNumber:=false;  inc(result); end;
  end;
  result:=result-2;
  try
    value:=StrToInt(StringReplace(number, '.', ',', [rfReplaceAll, rfIgnoreCase]));
  except
    on e:exception do begin
      value:=0;
      error(e.message);
    end;
  end;
end;

{
  extrahiert das nchste Token aus der Formel (in FFormula)
}
function TjkMathParser.getToken: tjkpToken;
var ch:char;
begin
  result.typ:=jkptEND;
  result.value:=ErrorD(0, 0);
  result.name:='';
  if FPosition<=FLength then begin
    ch:=FFormula[FPosition];
    //writeln('  -- '+ch);
    inc(FPosition);
    case ch of
      #13, #10, ';': result.typ:=jkptPRINT;
      '*': result.typ:=jkptMUL;
      '/': result.typ:=jkptDIV;
      '+': result.typ:=jkptPLUS;
      '-': result.typ:=jkptMINUS;
      '%': result.typ:=jkptMOD;
      '(': result.typ:=jkptLB;
      ')': result.typ:=jkptRB;
      '{': result.typ:=jkptLEB;
      '}': result.typ:=jkptREB;
      '=': result.typ:=jkptASSIGN;
      '^': result.typ:=jkptPOWER;
      ',': result.typ:=jkptARGUMENTDIVIDER;
      '|': result.typ:=jkptABSBrackets;
      '!': result.typ:=jkptFactorial;
      '0'..'9', '.': begin
        result.typ:=jkptNUMBER;
        result.value.error:=0;
        FPosition:=getNumber(FFormula, FPosition-1, result.value.value)+1;
        //writeln('    -> ',result.value, '   [', FPosition,']')
      end;
      '$': begin
        result.typ:=jkptNUMBER;
        result.value.error:=0;
        FPosition:=getHexNumber(FFormula, FPosition-1, result.value.value)+1;
      end;
      'a'..'z', '', '_': begin
        result.typ:=jkptNAME;
        FPosition:=getNAME(FFormula, FPosition-1, result.name)+1;
        if FPosition<=FLength then
          if FFormula[FPosition]='(' then begin
            result.typ:=jkptFUNCTION;
            inc(FPosition);
          end;
        //writeln('  -> ',result.value, '   [', FPosition,']')
      end;
    else
      if not(ch in jkpWhiteSpaces) then Error('Falsches Token  ("'+ch+'"  #'+inttostr(ord(ch))+')');
    end;
  end;
end;

// wertet die bergebene Formel aus.
function TjkMathParser.Parse(formula: string):ErrorDouble;
begin
  result:=ErrorD(0, 0);
  FNumberOfErrors:=0;
  FFormula:=FormatString(formula);
  FLength:=length(Fformula);
  FCurrentToken.typ:=jkptNone;
  FCurrentToken.value:=ErrorD(0, 0);
  FCurrentToken.name:='';

  if FLength>0 then begin
    FPosition:=1;
    while FPosition<=FLength do begin
      FCurrentToken:=getToken;
      if FCurrentToken.typ=jkptEnd then FPosition:=FLength+1 //Schleife beenden
      else if FCurrentToken.typ<>jkptPRINT then result:=Expression(false);
    end;
  end;
end;


//Auswerten von Zahlen, Konstanten, Funktionen, Absolutbetrag-Klammern (|...|)
function TjkMathParser.Primary(getTok: boolean): ErrorDouble;
var value, param1, param2:ErrorDouble;
    cname, fname, ferror:string;
    fdata: tjkpFunction;
begin
  result:=ErrorD(0,0);
  cname:='';
  fname:='';
  ferror:='';
  if getTok then FCurrentToken:=getToken;

  case FCurrentToken.typ of
    jkptNUMBER: begin
      result:=FCurrentToken.value;
      FCurrentToken:=getToken;
    end;
    jkptNAME: begin
      result:=GetConstant(FCurrentToken.name);
    end;
    jkptFUNCTION: begin
      fname:=FCurrentToken.name;
      //FCurrentToken:=getToken;
      fdata:=getFunction(fname);
      try
        if fdata.name<>'' then begin
          result:=ErrorD(0,0);
          if fdata.typ=jkpftSingle then begin
            param1:=Expression(true);
            result:=fdata.singleFunction(param1, ferror);
          end else if fdata.typ=jkpftDouble then begin
            param1:=Expression(true);
            if FCurrentToken.typ=jkptARGUMENTDIVIDER then begin
              FCurrentToken:=getToken;
              param2:=expression(true);
              result:=fdata.DoubleFunction(param1, param2, ferror);
            end else error('Die Funktion "'+fname+'" bentigt zwei Parameter');
          end else error('interner Fehler: Funktion "'+fname+'" ist nicht definiert');
          if ferror<>'' then begin
            error(ferror);
            result:=ErrorD(0,0);
          end;
          FCurrentToken:=getToken;
        end else begin
          result:=ErrorD(0,0);
          error('Funktion "'+fname+'" ist nicht definiert');
        end;
      except
        on E: Exception do error('interner Fehler: '+e.Message);
      end;
    end;
    jkptMINUS: result:=Mul(-1, Primary(true));
    jkptABSBrackets: begin
      result.value:=abs(Expression(true).value);
      FCurrentToken:=getToken;
    end;
    jkptLB: begin
      value:=Expression(true);
      if not(FCurrentToken.typ in[jkptRB{, jkptEND}]) then begin
        error(') erwartet');
        result:=ErrorD(0,0);
      end else begin
        result:=value;
        FCurrentToken:=getToken;
      end;
    end;
    jkptLEB: begin
      result:=getErrorNumber;
    end;
  else
    result:=ErrorD(0,0);
    error('Zahl oder Bezeichner (Primary) erwartet');
  end;

  if FCurrentToken.typ=jkptPower then begin
    result:=power(result, Primary(true));
  end else if FCurrentToken.typ=jkptFactorial then begin
    if isInteger(result.value) then begin

      // Fehlerbehandlung von Fakultt ???
      result.error:=0;


      if result.value>=0 then
        result.value:=Factorial(trunc(result.value))
      else error('Fakultt x! nur fr x>=0 definiert');
    end else error('Fakultt x! nur fr ganzzahlige x definiert');
  end;
end;

//Multiplikation, Division, Potenz
procedure TjkMathParser.SetConstant(name: string; value: ErrorDouble);
begin
  if FConstants.ItemExists(name) then
    ErrorDouble(FConstants[name]^):=value
  else AddConstant(name, value);
end;

function TjkMathParser.Term(getTok: boolean): ErrorDouble;
var left, value:ErrorDouble;
    loop:boolean;
begin
  loop:=true;
  left:=Primary(getTok);
  result:=ErrorD(0,0);

  while loop do begin
    case FCurrentToken.typ of
      jkptMUL: left:=Mul(left, Primary(true));
      jkptDIV: begin
        value:=Primary(true);
        if value.value<>0 then left:=Divide(left, value)
        else Error('Division durch 0');
      end;
      jkptMOD: begin
        value:=Primary(true);
        if isInteger(value.value) and isInteger(left.value) then begin

          //Fehlerbehandlung von Modulo ???
          result.error:=0;


          try
            left.value:=trunc(left.value) mod trunc(value.value)
          except
            on e:Exception do
              error(e.message);
          end;
        end else error('Modulo nur fr Ganzzahlwerte definiert');
      end;
    else
      result:=left;
      loop:=false;
    end;
  end;
end;


function TjkMathParser.GetConstant(cname: string): ErrorDouble;
var constfound:boolean;
begin
  constfound:=false;
  result:=ErrorD(0,0);
  if (cname='random')or(cname='rnd') then begin result.value:=Random; constfound:=true; end
    else if FConstants.ItemExists(FCurrentToken.name) then begin result:=ErrorDouble(FConstants[FCurrentToken.name]^); constfound:=true; end;
  FCurrentToken:=getToken;
  if FCurrentToken.typ=jkptASSIGN then begin
    constfound:=false;
    result:=Expression(true);
    if (not constfound) and assigned(FExternalSetConst) then FExternalSetConst(self, cname, result, constfound);
    if not constfound then begin
      SetConstant(cname, result);
      constfound:=true;
    end;
    //writeln('Variable erzeugt  '+cname);
    //writeln('    '+FConstants.commatext);
  end else if not constfound then begin
    if assigned(FExternalGetConst) then FExternalGetConst(self, cname, result, constfound);
  end;
  if not constfound then error('unbekannte Konstante "'+cname+'"');
end;

function TjkMathParser.IsInteger(value: extended): boolean;
begin
  result:=false;
  if trunc(value)=value then result:=true;
end;

function Factorial(value: int64): int64;
var i:longint;
begin
  result:=1;
  if value>1 then for i:=1 to value do
    result:=result*i;
end;


function TjkMathParser.getErrorNumber: ErrorDouble;
begin
  {if text[FPosition]='$' then FPosition:=gethexNumber(text, FPosition, value.value)
    else FPosition:=getNumber(text, FPosition, value.value);}
  result.value:=Expression(true).value;
  if FCurrentToken.typ=jkptARGUMENTDIVIDER then begin
    result.error:=Expression(true).value;
    FCurrentToken:=getToken;
  end else if FCurrentToken.typ=jkptREB then begin
    result.error:=0;
    FCurrentToken:=getToken;
  end else error('falsches Format fr Fehlerbehaftete Zahl');
end;

begin
  Randomize; //Zufallszahlen-Generator initialisieren
end.


